//=============================================================================
// RPGツクールMZ - ChangeMaxLevel.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc 最大レベルをコマンドから変更できるようになります。
 * @author emoriiin979
 *
 * @help ChangeMaxLevel.js
 *
 * 最大レベルをコマンドから変更できるようになります。
 *
 * @param variableId
 * @text 変数ID
 * @desc 最大レベルを格納する変数IDを指定してください。
 * @type variable
 *
 * @command changeMaxLevel
 * @text 最大レベル変更
 * @desc 最大レベルを変更します。
 *
 * @arg newMaxLevel
 * @text 最大レベル
 * @desc 変更後の最大レベルを指定してください。
 * 現在のレベルより大きい値が指定された場合は変更されません。
 * @default 99
 * @type number
 * @min 1
 * @max 99
 */

(() => {
    "use strict";

    const pluginName = "ChangeMaxLevel";
    const parameters = PluginManager.parameters(pluginName);

    const paramVariableId = parseInt(parameters["variableId"]);

    //=========================================================================
    // プラグインコマンド
    //=========================================================================

    // 最大レベル変更
    PluginManager.registerCommand(pluginName, "changeMaxLevel", args => {
        $gameVariables.setValue(paramVariableId, parseInt(args["newMaxLevel"]));
        DataManager.changeMaxLevel(paramVariableId);
    });

    //-------------------------------------------------------------------------
    // DataManager
    //
    // 最大レベルを変数の値で書き換えます。

    DataManager.loadGame = function(savefileId) {
        const saveName = this.makeSavename(savefileId);
        return StorageManager.loadObject(saveName).then(contents => {
            this.createGameObjects();
            this.extractSaveContents(contents);
            this.correctDataErrors();
            this.changeMaxLevel(paramVariableId);
            return 0;
        });
    };

    DataManager.changeMaxLevel = function(variableId) {
        const newMaxLevel = $gameVariables.value(variableId) || 99;
        // 一人でも指定の最大レベルを上回っている場合は変更しない
        for (const actor of $gameParty.allMembers()) {
            if (actor.level > newMaxLevel) {
                return;
            }
        }
        for (const actor of $gameParty.allMembers()) {
            actor.actor().maxLevel = newMaxLevel;
        }
    };

    //-------------------------------------------------------------------------
    // BattleManager
    //
    // 最大レベルに達した時は警告メッセージをバトル終了時に出すよう改変

    const _BattleManager_displayExp = BattleManager.displayExp;
    BattleManager.displayExp = function() {
        const actor = $gameActors.actor(1);
        if (actor.isMaxLevel()) {
            $gameMessage.add(actor.level === 99
                ? "\\c[18]最大レベル\\c[0]に到達しています。"
                : "現時点での\\c[18]最大レベル\\c[0]に到達しています。"
            );
            return;
        }
        _BattleManager_displayExp.call(this);
    };

    //-------------------------------------------------------------------------
    // Game_Actor
    //
    // 最大レベルに必要な経験値以上を取得できないようにします。

    Game_Actor.prototype.gainExp = function(exp) {
        const newExp = Math.min(
            this.currentExp() + Math.round(exp * this.finalExpRate()),
            this.expForLevel(this.actor().maxLevel)
        );
        this.changeExp(newExp, this.shouldDisplayLevelUp());
    };
})();
